// MovableThruster.cpp: Implementierung der Klasse CMovableThruster.
//
//////////////////////////////////////////////////////////////////////

#include <math.h>
#include <stdio.h>
#include "MovableThruster.h"

//////////////////////////////////////////////////////////////////////
// Konstruktion/Destruktion
//////////////////////////////////////////////////////////////////////

CMovableThruster::CMovableThruster(VESSEL* pVessel, VECTOR3 vecPos, 
	double dMaxThrust, double dISP, VECTOR3 vecMeshThrustDir, 
	double dMinAngle, double dMaxAngle, int iRotAxis,
	double dExhaustLength, double dExhaustWidth, double dExhaustDist)
{
	m_bShuttleBroken = false;
	m_pVessel = pVessel;
	m_vecPos = vecPos;
	m_dMaxThrust = dMaxThrust;
	m_dISP = dISP;
	m_vecMeshThrustDir = m_vecThrustDir = vecMeshThrustDir;
	m_dThrDirWant = m_dThrDir = m_dLastAnimDir = 0.0;
	m_dThrRotSpeed = 0.0;
	m_dMinAngle = dMinAngle;
	m_dMaxAngle = dMaxAngle;
	m_iRotAxis = iRotAxis;
	m_dExhaustLength = dExhaustLength;
	m_dExhaustWidth = dExhaustWidth;
	m_dExhaustDist = dExhaustDist;
	m_puiMeshGroups = NULL;
	m_iNumGroups = 0;
	m_iMeshID = 0;
	m_uiMode = MOVE_TO;
	m_uiExhID = -1;
	m_dMeshToAnimAngle = 0.0;
	m_vh = NULL;
	m_dAnimAngle = 0.0;
//	m_pVessel->RegisterAnimation();
}

CMovableThruster::~CMovableThruster()
{
//	m_pVessel->UnregisterAnimation();
	delete m_pMeshGroupRotate;
	if(m_puiMeshGroups)
		delete m_puiMeshGroups;
}

bool CMovableThruster::Timestep(double dSimtimeDelta)
{
	double dDeltaAngle = 0.0;
	double dDirection = m_dThrRotSpeed>=0.0?1.0:-1.0;
	bool bRet = false;

	if(m_bShuttleBroken)
		return false;
		
	switch(m_uiMode)
	{
	case MOVE_TO:
		if(m_dThrDirWant != m_dThrDir || m_dThrRotSpeed != 0.0)
		{
			if(m_dThrDirWant > m_dThrDir)
			{
				// determine wether we have to brake the rotation by calculating
				// at what angle we would stop if we started braking now with half the
				// possible decceleration and compare this to the wanted angle
				if(m_dThrDirWant <= m_dThrDir + dDirection * m_dThrRotSpeed * m_dThrRotSpeed / THRUSTROTACCEL)
					m_dThrRotSpeed = max(0.0, m_dThrRotSpeed - THRUSTROTACCEL * dSimtimeDelta); // deccelerate
				else
					m_dThrRotSpeed = min(MAX_ROT_SPEED, m_dThrRotSpeed + THRUSTROTACCEL * dSimtimeDelta); // accelerate
				dDeltaAngle = m_dThrRotSpeed * dSimtimeDelta;
				// to prevent the thruster from steadily moving slightly around the wanted
				// angle (because we only use a fixed acceleration above) 
				// we "nail" it to the wanted angle if it has gone beyond it but 
				// has deccelerated to 1/10 of the max rotation speed
				if(m_dThrDirWant - m_dThrDir < dDeltaAngle && m_dThrRotSpeed < MAX_ROT_SPEED/10)
				{
					dDeltaAngle = m_dThrDirWant - m_dThrDir;
					m_dThrRotSpeed = 0.0;
				}
			}
			else // same as above in the other direction
			{
				if(m_dThrDirWant >= m_dThrDir + dDirection*m_dThrRotSpeed*m_dThrRotSpeed/(THRUSTROTACCEL))
					m_dThrRotSpeed = min(0.0, m_dThrRotSpeed+THRUSTROTACCEL*dSimtimeDelta);
				else
					m_dThrRotSpeed = max(-MAX_ROT_SPEED, m_dThrRotSpeed-THRUSTROTACCEL*dSimtimeDelta);
				dDeltaAngle = m_dThrRotSpeed*dSimtimeDelta;
				if(m_dThrDirWant - m_dThrDir > dDeltaAngle && m_dThrRotSpeed > -MAX_ROT_SPEED/10)
				{
					dDeltaAngle = m_dThrDirWant - m_dThrDir;
					m_dThrRotSpeed = 0.0;
				}
			}
		}
		break;
	case MOVE_FORWARD: 
		// essentially the same as in MOVE_TO only that our limiting angle is the 
		// maximum angle instead of the wanted angle and we move the wanted angle 
		// along with the current angle (this implies that the wanted angle will 
		// stay where the user stops the manual rotation while the thruster itself
		// will shoot a bit over it deccelerating its current rotation speed and 
		// then come back to this position)
		if(m_dMaxAngle <= m_dThrDir + dDirection*m_dThrRotSpeed*m_dThrRotSpeed/(THRUSTROTACCEL))
			m_dThrRotSpeed = max(0.0, m_dThrRotSpeed-THRUSTROTACCEL*dSimtimeDelta);
		else
			m_dThrRotSpeed = min(MAX_ROT_SPEED, m_dThrRotSpeed+THRUSTROTACCEL*dSimtimeDelta);
		dDeltaAngle = m_dThrRotSpeed*dSimtimeDelta;
		if(m_dMaxAngle - m_dThrDir < dDeltaAngle && m_dThrRotSpeed < MAX_ROT_SPEED/10)
		{
			dDeltaAngle = m_dMaxAngle - m_dThrDir;
			m_dThrRotSpeed = 0.0;
		}
		m_dThrDirWant = m_dThrDir + dDeltaAngle; 
		break;
	case MOVE_BACKWARD:
		if(m_dMinAngle >= m_dThrDir + dDirection*m_dThrRotSpeed*m_dThrRotSpeed/(THRUSTROTACCEL))
			m_dThrRotSpeed = min(0.0, m_dThrRotSpeed+THRUSTROTACCEL*dSimtimeDelta);
		else
			m_dThrRotSpeed = max(-MAX_ROT_SPEED, m_dThrRotSpeed-THRUSTROTACCEL*dSimtimeDelta);
		dDeltaAngle = m_dThrRotSpeed*dSimtimeDelta;
		if(m_dMinAngle - m_dThrDir > dDeltaAngle && m_dThrRotSpeed > -MAX_ROT_SPEED/10)
		{
			dDeltaAngle = m_dMinAngle - m_dThrDir;
			m_dThrRotSpeed = 0.0;
		}
		m_dThrDirWant = m_dThrDir + dDeltaAngle; 
		break;
	}

	if(dDeltaAngle != 0.0)	
	{
		m_dThrDir += dDeltaAngle;
		CalcThrustDirVec();
		m_pVessel->SetThrusterDir (m_th, m_vecThrustDir);
		bRet = true;
	}
	if((m_dLastAnimDir != m_dThrDir) && !oapiCameraInternal() 
		&& oapiCameraTarget() == m_pVessel->GetHandle())
	{
		m_pVessel->SetAnimation(m_uiAnimSeq, (m_dThrDir-m_dMinAngle+m_dAnimAngle)/((m_dMaxAngle-m_dMinAngle)*2));
		m_dLastAnimDir = m_dThrDir;
	}
	return bRet;
}


THRUSTER_HANDLE CMovableThruster::Create(PROPELLANT_HANDLE ph, SURFHANDLE shExhaustTex /*= NULL*/)
{
	m_shExhaustTex = shExhaustTex;
	m_ph = ph;
	m_th = m_pVessel->CreateThruster (m_vecPos, m_vecMeshThrustDir, m_dMaxThrust, m_ph, m_dISP);	
	AddExhaust();
	return m_th;
}

void CMovableThruster::RegisterAnimation(UINT* puiMeshGroups, int iNumGroups, int iMesh)
{
	m_puiMeshGroups = new UINT[iNumGroups];
	for(int i=0;i<iNumGroups;i++)
		m_puiMeshGroups[i] = puiMeshGroups[i];
	m_iNumGroups = iNumGroups;
	m_iMeshID = iMesh;
	m_dWholeAngle = (m_dMaxAngle - m_dMinAngle) * 2;

	m_uiAnimSeq = m_pVessel->CreateAnimation (-m_dMinAngle / m_dWholeAngle);
	m_pMeshGroupRotate = new MGROUP_ROTATE(iMesh, m_puiMeshGroups, 
		m_iNumGroups, m_vecPos, 
		_V(m_iRotAxis==X?1.0:0.0, m_iRotAxis==Y?1.0:0.0, m_iRotAxis==Z?1.0:0.0),
		(float) m_dWholeAngle);

	m_hAnimComp =
		m_pVessel->AddAnimationComponent (m_uiAnimSeq, 0.0, 1.0, m_pMeshGroupRotate);
	
/*	m_anmcRot.grp = m_puiMeshGroups;
	m_anmcRot.ngrp = m_iNumGroups;
	m_anmcRot.state0 = 0.0;
	m_anmcRot.state1 = 1.0;
	m_anmcRot.trans.P.rotparam.ref = m_vecPos;
	m_anmcRot.trans.P.rotparam.axis = _V(m_iRotAxis==X?1.0:0.0, m_iRotAxis==Y?1.0:0.0, m_iRotAxis==Z?1.0:0.0);
	m_anmcRot.trans.P.rotparam.angle = (float) dWholeAngle;
	m_anmcRot.trans.nmesh = iMesh;
	m_anmcRot.trans.ngrp = 0;
	m_anmcRot.trans.transform = MESHGROUP_TRANSFORM::ROTATE;

	m_uiAnimSeq = m_pVessel->RegisterAnimSequence (-m_dMinAngle / dWholeAngle);
	m_pVessel->AddAnimComp (m_uiAnimSeq, &m_anmcRot);*/
}

void CMovableThruster::AddExhaust()
{
	if(m_uiExhID >= 0)
		m_pVessel->DelExhaust(m_uiExhID);
	m_uiExhID = m_pVessel->AddExhaust (m_th, m_dExhaustLength, m_dExhaustWidth, m_dExhaustDist, m_shExhaustTex);
}

void CMovableThruster::SetThrustDirWant(double dDirWant)
{
	m_dThrDirWant = max(min(dDirWant, m_dMaxAngle), m_dMinAngle);
}

THRUSTER_HANDLE CMovableThruster::GetThrHandle()
{
	return m_th;
}

double CMovableThruster::GetThrustDir()
{
	return m_dThrDir;
}

double CMovableThruster::GetThrustDirWant()
{
	return m_dThrDirWant;
}

void CMovableThruster::ForceThrustDir(double dThrDir)
{
	m_dThrDir = max(min(dThrDir, m_dMaxAngle), m_dMinAngle);
	m_dThrDirWant = m_dThrDir; 
	m_dThrRotSpeed = 0.0;
}

void CMovableThruster::SetThrustLevel(double dThrLevel)
{
	m_pVessel->SetThrusterLevel(m_th, dThrLevel);
}

double CMovableThruster::GetThrustLevel()
{
	return m_pVessel->GetThrusterLevel(m_th);
}

void CMovableThruster::PostCreation()
{
	CalcThrustDirVec();
	m_pVessel->SetThrusterDir (m_th, m_vecThrustDir);
	m_pVessel->SetAnimation(m_uiAnimSeq, (m_dThrDir-m_dMinAngle+m_dAnimAngle)/((m_dMaxAngle-m_dMinAngle)*2));
	m_dLastAnimDir = m_dThrDir;
}

void CMovableThruster::CalcThrustDirVec()
{
	double dSin = sin(m_dThrDir);
	double dCos = cos(m_dThrDir);
	switch(m_iRotAxis)
	{
	case X:
		m_vecThrustDir = _V(m_vecMeshThrustDir.x, m_vecMeshThrustDir.y*dCos - m_vecMeshThrustDir.z*dSin, m_vecMeshThrustDir.y*dSin + m_vecMeshThrustDir.z*dCos);
		break;
	case Y:
		m_vecThrustDir = _V(m_vecMeshThrustDir.x*dCos + m_vecMeshThrustDir.z*dSin, m_vecMeshThrustDir.y, -m_vecMeshThrustDir.x*dSin + m_vecMeshThrustDir.z*dCos);
		break;
	case Z:
		m_vecThrustDir = _V(m_vecMeshThrustDir.x*dCos-m_vecMeshThrustDir.y*dSin, -m_vecMeshThrustDir.x*dSin+m_vecMeshThrustDir.y*dCos, m_vecMeshThrustDir.z);
		break;
	}
}

void CMovableThruster::AddThrustDirWant(double dDirDelta)
{
	m_dThrDirWant = max(min(m_dThrDirWant + dDirDelta, m_dMaxAngle), m_dMinAngle);
}

void CMovableThruster::RotateForward()
{
	m_uiMode = MOVE_FORWARD;
}

void CMovableThruster::RotateBackward()
{
	m_uiMode = MOVE_BACKWARD;
}

void CMovableThruster::RotateStop()
{
	m_uiMode = MOVE_TO;
}



void CMovableThruster::ChangePosAndDir(VECTOR3 vecPos, VECTOR3 vecMeshThrustDir)
{
	m_vecMeshThrustDir = vecMeshThrustDir;
	m_vecPos = vecPos;
	m_pVessel->SetThrusterRef (m_th, m_vecPos);
	CalcThrustDirVec();
	m_pVessel->SetThrusterDir (m_th, m_vecThrustDir);
}

void CMovableThruster::SetExhaustParams(double dLength, double dWidth, double dDist)
{
	m_dExhaustLength = dLength;
	m_dExhaustWidth = dWidth;
	m_dExhaustDist = dDist;
	AddExhaust();
}

void CMovableThruster::MeshChanged(int iNewMeshID)
{
	m_dAnimAngle = m_dLastAnimDir;
	m_dLastAnimDir = 0.0;
	m_iMeshID = iNewMeshID;

	m_pVessel->DelAnimationComponent(m_uiAnimSeq, m_hAnimComp);
	
	delete m_pMeshGroupRotate;
	m_pMeshGroupRotate = 
		new MGROUP_ROTATE(iNewMeshID, m_puiMeshGroups, m_iNumGroups, m_vecPos, 
		_V(m_iRotAxis==X?1.0:0.0, m_iRotAxis==Y?1.0:0.0, m_iRotAxis==Z?1.0:0.0),
		(float) m_dWholeAngle);

	m_hAnimComp =
		m_pVessel->AddAnimationComponent (m_uiAnimSeq, 0.0, 1.0, m_pMeshGroupRotate);
}

void CMovableThruster::RotateNewMeshToThrDir()
{
/*	bool bBla;
 
	if(m_dMeshToAnimAngle != 0.0 && m_vh != NULL)
	{
		MESHGROUP_TRANSFORM mtRot;
		mtRot.P.rotparam.ref = m_vecPos;
		mtRot.P.rotparam.axis = _V(m_iRotAxis==X?1.0:0.0, m_iRotAxis==Y?1.0:0.0, m_iRotAxis==Z?1.0:0.0);
		mtRot.P.rotparam.angle = (float) m_dThrDir;
		m_mtRot.nmesh = 0;//m_iMeshID;
		m_mtRot.transform = MESHGROUP_TRANSFORM::ROTATE;
		for(int iIndex = 0;iIndex < m_iNumGroups;iIndex++)
		{
			mtRot.ngrp = m_puiMeshGroups[iIndex];
			bBla = m_pVessel->MeshgroupTransform(m_vh, mtRot);
		}
		m_dMeshToAnimAngle = 0.0;
	}*/
}

void CMovableThruster::VisualCreated(VISHANDLE vh)
{
	m_vh = vh;
	Animate(oapiGetSimTime());
}

void CMovableThruster::VisualDestroyed(VISHANDLE vh)
{
	m_vh = NULL;
}

void CMovableThruster::Animate(double dSimTime)
{
/*	if(m_bFirstAnimate)
	{
		m_bFirstAnimate = false;
		return;
	}

	if(m_vh && m_dThrDir != m_dCurDir)
	{
		m_mtRot.P.rotparam.angle = (float) (m_dThrDir - m_dCurDir);
		m_mtRot.nmesh = m_iMeshID;
		for(int iIndex = 0;iIndex < m_iNumGroups;iIndex++)
		{
			m_mtRot.ngrp = m_puiMeshGroups[iIndex];
			if(!m_pVessel->MeshgroupTransform(m_vh, m_mtRot))
				sprintf(oapiDebugString(), "FUCK");
		}
		m_dCurDir = m_dThrDir;
	}*/
}

VECTOR3 CMovableThruster::GetThrustDirVec()
{
	return m_vecThrustDir; 
}

VECTOR3 CMovableThruster::GetPos()
{
	return m_vecPos;
}

void CMovableThruster::BreakUp()
{
	m_bShuttleBroken = true;
}
