// FuelManagement.cpp: Implementation of Class CFuelManagement.
//
//////////////////////////////////////////////////////////////////////

#include "FuelManagement.h"
#include "ShuttleA.h"
#include "stdio.h"

CFuelManagement::CFuelManagement()
{
	for(int iIndex=EXT1;iIndex<NUMTANKS;iIndex++)
	{
		m_arrTanks[iIndex].m_bPresent = false;
		m_arrTanks[iIndex].m_dCurFuel = EXT_TANK_FUEL;
		m_arrTanks[iIndex].m_dMaxFuel = EXT_TANK_FUEL;
		m_arrTanks[iIndex].m_dLastFuel = EXT_TANK_FUEL;
		m_arrTanks[iIndex].m_dCurFlowRate = 0.0;
	}

	m_arrTanks[INT].m_dLastFuel = m_arrTanks[INT].m_dCurFuel = m_arrTanks[INT].m_dMaxFuel = INT_TANK_FUEL;

	m_arrTanks[RCS].m_dLastFuel = m_arrTanks[RCS].m_dCurFuel = m_arrTanks[RCS].m_dMaxFuel = MAX_RCS_FUEL;

	for(iIndex = 0;iIndex < NUMVALVES;iIndex++)
		m_arriValveStates[iIndex] = VLVST_OFF;

	m_FuelPump.m_iMode = FPMODE_OFF;
	m_FuelPump.m_fActLvl = 0.0;
	m_FuelPump.m_fCurFlowRate = 0.0;
	m_FuelPump.m_fTargetLvl = 0.0;
}

CFuelManagement::~CFuelManagement()
{

}

void CFuelManagement::Init(int *iCargoConf, VESSEL* pVessel)
{
	m_pVessel = pVessel;
	int iIndex, iNumExtTanks = 0;
	//external tanks
	for(iIndex=EXT1;iIndex<=EXT4;iIndex++)
	{
		if(iCargoConf[iIndex] == ShuttleA::TANK)
		{
			m_arrTanks[iIndex].m_bPresent = true;
			iNumExtTanks++;
			m_arrTanks[iIndex].m_ph = m_pVessel->CreatePropellantResource (EXT_TANK_FUEL);
			m_pVessel->SetPropellantMass(m_arrTanks[iIndex].m_ph, m_arrTanks[iIndex].m_dCurFuel);
		}
		else
		{
			m_arrTanks[iIndex].m_bPresent = false;
			m_arrTanks[iIndex].m_dCurFuel = 0.0;
			m_arrTanks[iIndex].m_ph = NULL;
			m_arriValveStates[VLV_EXT1 + iIndex] = VLVST_DISABLED;
		}
	}

	m_arrTanks[INT].m_bPresent = true;
	m_arrTanks[INT].m_ph = m_pVessel->CreatePropellantResource (INT_TANK_FUEL); // main tank
	m_pVessel->SetPropellantMass(m_arrTanks[INT].m_ph, m_arrTanks[INT].m_dCurFuel);

	m_arrTanks[RCS].m_bPresent = true;
	m_arrTanks[RCS].m_ph = m_pVessel->CreatePropellantResource (MAX_RCS_FUEL);  // RCS tank
	m_pVessel->SetPropellantMass(m_arrTanks[RCS].m_ph, m_arrTanks[RCS].m_dCurFuel);

}

void CFuelManagement::ResetValveState(int iValveState, int iFromValve, int iToValve /*=NUM_VALVES-1*/)
{
	int iValve;
	for(iValve = iFromValve;iValve <= iToValve;iValve++)
	{
		if(GetValveState(iValve) == iValveState)
			SetValveState(iValve, VLVST_OFF);
	}
}	

int CFuelManagement::CheckValveState(int iValveState, int iFromValve, int iToValve /*=NUM_VALVES-1*/) const
{
	int iValve;
	for(iValve = iFromValve;iValve <= iToValve;iValve++)
	{
		if(GetValveState(iValve) == iValveState)
			return iValve;
	}
	return -1;

}

void CFuelManagement::ToggleValve(int iValve)
{
	switch(GetValveState(iValve))
	{
	case VLVST_OFF:
		if(iValve <= VLV_MAIN)
		{
			if(iValve >= VLV_HOVER)
			{
				ResetValveState(VLVST_DST, VLV_EXT1);
				ResetValveState(VLVST_SRC, VLV_RCSPIPE);
			}
			SetValveState(iValve, VLVST_DST);
		}
		else
		{
			if(iValve >= VLV_EXT1 && iValve <= VLV_EXT4 && !m_arrTanks[iValve-VLV_EXT1+EXT1].m_bPresent)
				return;
			if(iValve >= VLV_RCSPIPE)
				ResetValveState(VLVST_DST, VLV_HOVER, VLV_MAIN);
			if(CheckValveState(VLVST_SRC, VLV_EXT1) != -1)
			{
				if(CheckValveState(VLVST_DST, VLV_HOVER, VLV_MAIN) != -1)
				{
					ResetValveState(VLVST_SRC, VLV_EXT1);
					SetValveState(iValve, VLVST_SRC);
				}
				else
				{
					ResetValveState(VLVST_DST, VLV_EXT1);
					SetValveState(iValve, VLVST_DST);
				}
			}
			else
			{
				ResetValveState(VLVST_SRC, VLV_EXT1);
				SetValveState(iValve, VLVST_SRC);
				if(iValve >= VLV_RCSPIPE)
					ResetValveState(VLVST_DST, VLV_HOVER, VLV_MAIN);
			}
		}
		break;
	case VLVST_SRC:
		if(iValve <= VLV_MAIN)
			SetValveState(iValve, VLVST_OFF);
		else
		{
			ResetValveState(VLVST_DST, VLV_HOVER);
			SetValveState(iValve, VLVST_DST);
		}
		break;
	case VLVST_DST:
		SetValveState(iValve, VLVST_OFF);
		break;
	}
}

void CFuelManagement::Timestep(double dSimTime)
{
	ShuttleA* pSh = (ShuttleA*)m_pVessel;
	if(pSh->m_bShuttleBroken)
		return;

	int iTank;
	double dSimStep = oapiGetSimStep();
	STank* pTank;
	bool bExtFuelPrs = false;
	bool bSetTgtLvl = false;
	bool bSrcPHValid, bDstPHValid;
	double dMaxSrcFlow, dMaxDstFlow, dSrcPropMass, dDstPropMass, dDstPropMaxMass, dActFlow;
	PROPELLANT_HANDLE phSrc, phDst;

	if(pSh->DockingStatus(0))
	{
		if(GetValveState(VLV_DOCK) == VLVST_DISABLED)
			SetValveState(VLV_DOCK, VLVST_OFF);
	}
	else 
	{
		if(GetValveState(VLV_DOCK) != VLVST_DISABLED)
			SetValveState(VLV_DOCK, VLVST_DISABLED);
	}

	if(pSh->IsLanded())
		bExtFuelPrs = true;

	for(iTank = EXT1;iTank <= RCS;iTank++)
	{
		pTank = &m_arrTanks[iTank];
		if(pTank->m_bPresent)
		{
			pTank->m_dCurFuel = pSh->GetPropellantMass(pTank->m_ph);

			if((bExtFuelPrs && pTank->m_dCurFuel != pTank->m_dLastFuel) && (pTank->m_dCurFuel == pTank->m_dMaxFuel))
			{
				pTank->m_dCurFuel = pTank->m_dLastFuel;
				pSh->SetPropellantMass(pTank->m_ph, pTank->m_dLastFuel);
			}
		}
	}

	switch(m_FuelPump.m_iMode)
	{
	case FPMODE_OFF:
		m_FuelPump.m_fTargetLvl = 0.0;
		FuelPumpSpeedChange(dSimStep);
		m_FuelPump.m_fCurFlowRate = 0.0;
		break;
	case FPMODE_AUTO:
		iTank = CheckValveState(VLVST_SRC, VLV_EXT1, VLV_INT) - VLV_EXT1 + EXT1;
		m_FuelPump.m_fTargetLvl = m_arrTanks[iTank].m_dCurFlowRate / FUELPUMP_100PCNT_FLOWRATE;
		FuelPumpSpeedChange(dSimStep);
		m_FuelPump.m_fCurFlowRate = m_arrTanks[iTank].m_dCurFlowRate;
		break;
	case FPMODE_MAN:
		bSrcPHValid = GetPropHandleFromValve(CheckValveState(VLVST_SRC, VLV_EXT1), phSrc);
		bDstPHValid = GetPropHandleFromValve(CheckValveState(VLVST_DST, VLV_EXT1), phDst);
		FuelPumpSpeedChange(dSimStep);
		if(phSrc == NULL) // refueling from Hatch ?
		{
			if(bExtFuelPrs && bSrcPHValid)
				dMaxSrcFlow = FUELPUMP_100PCNT_FLOWRATE * 2 * dSimStep;
			else
			{
				dMaxSrcFlow = 0.0;
				bSetTgtLvl = true;
			}
		}
		else 
		{
			dMaxSrcFlow = m_FuelPump.m_fActLvl * FUELPUMP_100PCNT_FLOWRATE * dSimStep;
			dSrcPropMass = pSh->GetPropellantMass(phSrc);
			if(dSrcPropMass < dMaxSrcFlow)
			{
				dMaxSrcFlow = dSrcPropMass;
				bSetTgtLvl = true;
			}
		}
		if(phDst == NULL) // venting fuel
		{
			if(bDstPHValid)
				dMaxDstFlow = FUELPUMP_100PCNT_FLOWRATE * 2 * dSimStep;
			else 
			{
				dMaxDstFlow = 0.0;
				bSetTgtLvl = true;
			}
		}
		else
		{
			dMaxDstFlow = m_FuelPump.m_fActLvl * FUELPUMP_100PCNT_FLOWRATE * dSimStep;
			dDstPropMass = pSh->GetPropellantMass(phDst);
			dDstPropMaxMass = pSh->GetPropellantMaxMass(phDst);
			if((dDstPropMaxMass - dDstPropMass) < dMaxDstFlow)
			{
				dMaxDstFlow = (dDstPropMaxMass - dDstPropMass);
				bSetTgtLvl = true;
			}
		}
		dActFlow = min(dMaxSrcFlow, dMaxDstFlow);
		m_FuelPump.m_fCurFlowRate = dActFlow / dSimStep;
		if(bSetTgtLvl)
			m_FuelPump.m_fTargetLvl = m_FuelPump.m_fCurFlowRate / FUELPUMP_100PCNT_FLOWRATE;
		if(phSrc != NULL)
			pSh->SetPropellantMass(phSrc, dSrcPropMass - dActFlow);
		if(phDst != NULL)
			pSh->SetPropellantMass(phDst, dDstPropMass + dActFlow);
		break;
	}

	for(iTank = EXT1;iTank <= RCS;iTank++)
	{
		pTank = &m_arrTanks[iTank];
		if(pTank->m_bPresent)
		{
			pTank->m_dLastFuel = pTank->m_dCurFuel = pSh->GetPropellantMass(pTank->m_ph);
			pTank->m_dCurFlowRate = pSh->GetPropellantFlowrate(m_arrTanks[iTank].m_ph); 
		}
	}
}

void CFuelManagement::UpdateThrMainTankConnections()
{
	ShuttleA* pSh = (ShuttleA*) m_pVessel;
	
	int iValve, iTank = -1;
	if((iValve = CheckValveState(VLVST_SRC, VLV_EXT1, VLV_RCSPIPE)) != -1)
		iTank = iValve - VLV_EXT1 + EXT1;
	if(pSh->m_bShuttleBroken)
		iTank = -1;

	if(HasFuelPumpConnection(pSh->th_hover[0]) && iTank != -1)
		pSh->SetThrusterResource(pSh->th_hover[0], m_arrTanks[iTank].m_ph);
	else
		pSh->SetThrusterResource(pSh->th_hover[0], NULL);
	
	if(HasFuelPumpConnection(pSh->th_hover[1]) && iTank != -1)
		pSh->SetThrusterResource(pSh->th_hover[1], m_arrTanks[iTank].m_ph);
	else
		pSh->SetThrusterResource(pSh->th_hover[1], NULL);

	if(HasFuelPumpConnection(pSh->th_pod[1]) && iTank != -1)
		pSh->SetThrusterResource(pSh->th_pod[1], m_arrTanks[iTank].m_ph);
	else
		pSh->SetThrusterResource(pSh->th_pod[1], NULL);

	if(HasFuelPumpConnection(pSh->th_pod[0]) && iTank != -1)
	{
		pSh->SetThrusterResource(pSh->th_pod[0], m_arrTanks[iTank].m_ph);
		if(pSh->m_bAuxPodDmg)
			m_pVessel->SetThrusterLevel(pSh->th_pod[0], 1.0);
	}
	else
		pSh->SetThrusterResource(pSh->th_pod[0], NULL);

	if(HasFuelPumpConnection(pSh->th_main[0]) && iTank != -1)
		pSh->SetThrusterResource(pSh->th_main[0], m_arrTanks[iTank].m_ph);
	else
		pSh->SetThrusterResource(pSh->th_main[0], NULL);

	if(HasFuelPumpConnection(pSh->th_main[1]) && iTank != -1)
		pSh->SetThrusterResource(pSh->th_main[1], m_arrTanks[iTank].m_ph);
	else
		pSh->SetThrusterResource(pSh->th_main[1], NULL);
}


void CFuelManagement::UpdateThrRCSTankConnections()
{
	int iIndex;
	ShuttleA* pSh = (ShuttleA*) m_pVessel;
	PROPELLANT_HANDLE ph;
	if(!pSh->m_bShuttleBroken && GetValveState(VLV_RCS) == VLVST_DST)
		ph = m_arrTanks[RCS].m_ph;
	else
		ph = NULL;

	for(iIndex = 0;iIndex<14;iIndex++)
		pSh->SetThrusterResource(pSh->m_arrthAtt[iIndex], ph);
}

double CFuelManagement::GetTankLvlNormalized(int iTank)
{
	return m_arrTanks[iTank].m_dCurFuel / m_arrTanks[iTank].m_dMaxFuel;
}

void CFuelManagement::SaveValveStates(char * pSaveString)
{
	
	for(int iValve=0;iValve < NUMVALVES;iValve++)
	{
		pSaveString[iValve] = GetValveState(iValve) + 0x30;
	}
	pSaveString[iValve] = 0;
}

void CFuelManagement::SetTankLvlNormalized(int iTank, double dLvl)
{
	m_arrTanks[iTank].m_dLastFuel = m_arrTanks[iTank].m_dCurFuel = m_arrTanks[iTank].m_dMaxFuel * dLvl;

}

void CFuelManagement::LoadValveStates(char * pLoadString)
{
	bool bEOLFound = false;
	for(int iValve=0;iValve < NUMVALVES;iValve++)
	{
		if(pLoadString[iValve] == 0)
			bEOLFound = true;
		if(!bEOLFound && pLoadString[iValve] >= 0x30 && pLoadString[iValve] <= 0x33)
			m_arriValveStates[iValve] = pLoadString[iValve] - 0x30;
		else
			m_arriValveStates[iValve] = VLVST_OFF;

		if(iValve <= VLV_MAIN && m_arriValveStates[iValve] == VLVST_SRC || m_arriValveStates[iValve] == VLVST_DISABLED)
			m_arriValveStates[iValve] = VLVST_OFF;
	}
}

void CFuelManagement::UpdateTankConnections()
{
	UpdateThrMainTankConnections();
	UpdateThrRCSTankConnections();

	int iSrc = CheckValveState(VLVST_SRC, VLV_EXT1);
	int iDst = CheckValveState(VLVST_DST, VLV_HOVER);

	if(iSrc != -1 && iDst != -1)
	{
		if(iDst <= VLV_MAIN)
		{
			m_FuelPump.m_fTargetLvl = 0.05f;
			m_FuelPump.m_iMode = FPMODE_AUTO;
		}
		else
			m_FuelPump.m_iMode = FPMODE_MAN;
	}
	else
		m_FuelPump.m_iMode = FPMODE_OFF;
}

bool CFuelManagement::HasFuelPumpConnection(THRUSTER_HANDLE th) const
{
	ShuttleA* pSh = (ShuttleA*) m_pVessel;
	if(pSh->m_bShuttleBroken)
		return false;

	if(th == pSh->th_hover[0] && GetValveState(VLV_FRONT_HOVER) == VLVST_DST && GetValveState(VLV_HOVER) == VLVST_DST)	
		return true;

	if(th == pSh->th_hover[1] && GetValveState(VLV_REAR_HOVER) == VLVST_DST && GetValveState(VLV_HOVER) == VLVST_DST)
		return true;

	if(th == pSh->th_pod[1] && GetValveState(VLV_RIGHT_AUX) == VLVST_DST && GetValveState(VLV_AUX) == VLVST_DST)
		return true;

	if(th == pSh->th_pod[0] && GetValveState(VLV_LEFT_AUX) == VLVST_DST && GetValveState(VLV_AUX) == VLVST_DST)
		return true;

	if(th == pSh->th_main[0] && GetValveState(VLV_LEFT_MAIN) == VLVST_DST && GetValveState(VLV_MAIN) == VLVST_DST)
		return true;

	if(th == pSh->th_main[1] && GetValveState(VLV_RIGHT_MAIN) == VLVST_DST && GetValveState(VLV_MAIN) == VLVST_DST)
		return true;

	return false;
}

double CFuelManagement::GetTankFuelMass(int iTank)
{
	return m_arrTanks[iTank].m_dCurFuel;
}

void CFuelManagement::BreakUp()
{
//	SetMainTankFuelPrs(GetActiveTanks() != 0);
	for(int iTank = EXT2;iTank <= INT;iTank++)
	{
		m_arrTanks[iTank].m_bPresent = false;
		m_arrTanks[iTank].m_dLastFuel = m_arrTanks[iTank].m_dCurFuel = 0;
		m_pVessel->DelPropellantResource(m_arrTanks[iTank].m_ph);
		SetValveState(iTank - EXT1 + VLV_EXT1, VLVST_DISABLED);
	}
	m_FuelPump.m_fActLvl = 0.0;
	m_FuelPump.m_fCurFlowRate = 0.0;
	m_FuelPump.m_iMode = FPMODE_OFF;
	UpdateThrMainTankConnections();
	UpdateThrRCSTankConnections();
}

bool CFuelManagement::HasTankConnectionRearPart(THRUSTER_HANDLE th) const
{
	ShuttleA* pSh = (ShuttleA*) m_pVessel;

	if(CheckValveState(VLVST_SRC, VLV_EXT3, VLV_INT) == -1)
		return false;

	if(th == pSh->th_hover[1] || th == pSh->th_pod[1] || th == pSh->th_pod[0] 
		|| th == pSh->th_main[0] || th == pSh->th_main[1])
		return HasFuelPumpConnection(th);

	return false;
}

void CFuelManagement::GetBrokenFuelMasses(double& dConnected, double& dDisconnected)
{
	bool bValve;
	int iTank;
	STank* pTank;
	
	dConnected = dDisconnected = 0;

	for(iTank = EXT3;iTank <= INT;iTank++)
	{
		pTank = &m_arrTanks[iTank];
		bValve = (GetValveState(VLV_EXT1 + iTank) == VLVST_SRC);
		if(pTank->m_bPresent && bValve)
			dConnected += pTank->m_dCurFuel;
		else 
			dDisconnected += pTank->m_dCurFuel;
	}
}


int CFuelManagement::GetValveState(int iValve) const
{
	return m_arriValveStates[iValve];
}

void CFuelManagement::SetValveState(int iValve, int iValveState)
{
	m_arriValveStates[iValve] = iValveState;
	UpdateTankConnections();
}

void CFuelManagement::FuelPumpSpeedChange(double dSimStep)
{
	if(m_FuelPump.m_fActLvl == m_FuelPump.m_fTargetLvl)
		return;
	if(fabs(m_FuelPump.m_fActLvl - m_FuelPump.m_fTargetLvl) <= dSimStep * FUELPUMP_CHGSPEED)
		m_FuelPump.m_fActLvl = m_FuelPump.m_fTargetLvl;
	else
		m_FuelPump.m_fActLvl = m_FuelPump.m_fActLvl + 
		(dSimStep * FUELPUMP_CHGSPEED * ((m_FuelPump.m_fTargetLvl>m_FuelPump.m_fActLvl)?1.0:-1.0));
}

bool CFuelManagement::GetPropHandleFromValve(int iValve, PROPELLANT_HANDLE& ph)
{
	OBJHANDLE objDockedVessel;
	ShuttleA* pSh = (ShuttleA*) m_pVessel;

	if(iValve == VLV_HATCHVENT)
	{
		ph = NULL;
		return true;
	}

	if(iValve >= VLV_EXT1 && iValve <= VLV_INT)
	{
		ph = m_arrTanks[iValve-VLV_EXT1+EXT1].m_ph;
	}
	if(iValve == VLV_RCSPIPE)
	{
		ph = m_arrTanks[RCS].m_ph;
	}
	if(iValve == VLV_DOCK)
	{
		objDockedVessel = pSh->GetDockStatus(pSh->GetDockHandle(0));
		if(objDockedVessel == NULL)
			ph = NULL;
		else
			ph = oapiGetPropellantHandle (objDockedVessel, 0);
	}
	if(ph == NULL) 
		return false;
	else
		return true;
}
